using System;

namespace Server.Items
{
	public abstract class BaseSilkOuterTorso : BaseClothing
	{
		public BaseSilkOuterTorso( int itemID ) : this( itemID, 0 )
		{
		}

		public BaseSilkOuterTorso( int itemID, int hue ) : base( itemID, Layer.OuterTorso, hue )
		{
		}

        public BaseSilkOuterTorso(Serial serial)
            : base(serial)
		{
		}

		public override void Serialize( GenericWriter writer )
		{
			base.Serialize( writer );

			writer.Write( (int) 0 ); // version
		}

		public override void Deserialize( GenericReader reader )
		{
			base.Deserialize( reader );

			int version = reader.ReadInt();
		}
	}

	[Flipable( 0x230E, 0x230D )]
	public class GildedSilkDress : BaseSilkOuterTorso
	{
        public override int BasePhysicalResistance { get { return 2; } }
        public override int BaseFireResistance { get { return 1; } }
        public override int BaseColdResistance { get { return 4; } }
        public override int BasePoisonResistance { get { return 4; } }
        public override int BaseEnergyResistance { get { return 1; } }

        public override int InitMinHits { get { return 10; } }
        public override int InitMaxHits { get { return 50; } }

		[Constructable]
		public GildedSilkDress() : this( 0 )
		{
		}

		[Constructable]
		public GildedSilkDress( int hue ) : base( 0x230E, hue )
		{
			Weight = 3.0;
            Name = "Gilded Silk Dress";
		}

		public GildedSilkDress( Serial serial ) : base( serial )
		{
		}

		public override void Serialize( GenericWriter writer )
		{
			base.Serialize( writer );

			writer.Write( (int) 0 ); // version
		}

		public override void Deserialize( GenericReader reader )
		{
			base.Deserialize( reader );

			int version = reader.ReadInt();
		}
	}

	[Flipable( 0x1F00, 0x1EFF )]
	public class FancySilkDress : BaseOuterTorso
	{
        public override int BasePhysicalResistance { get { return 2; } }
        public override int BaseFireResistance { get { return 1; } }
        public override int BaseColdResistance { get { return 4; } }
        public override int BasePoisonResistance { get { return 4; } }
        public override int BaseEnergyResistance { get { return 1; } }

        public override int InitMinHits { get { return 10; } }
        public override int InitMaxHits { get { return 50; } }

		[Constructable]
		public FancySilkDress() : this( 0 )
		{
		}

		[Constructable]
		public FancySilkDress( int hue ) : base( 0x1F00, hue )
		{
			Weight = 3.0;
            Name = "Fancy Silk Dress";
		}

        public FancySilkDress(Serial serial)
            : base(serial)
		{
		}

		public override void Serialize( GenericWriter writer )
		{
			base.Serialize( writer );

			writer.Write( (int) 0 ); // version
		}

		public override void Deserialize( GenericReader reader )
		{
			base.Deserialize( reader );

			int version = reader.ReadInt();
		}
	}

	[Flipable]
	public class SilkRobe : BaseSilkOuterTorso, IArcaneEquip
	{
		#region Arcane Impl
		private int m_MaxArcaneCharges, m_CurArcaneCharges;

		[CommandProperty( AccessLevel.GameMaster )]
		public int MaxArcaneCharges
		{
			get{ return m_MaxArcaneCharges; }
			set{ m_MaxArcaneCharges = value; InvalidateProperties(); Update(); }
		}

		[CommandProperty( AccessLevel.GameMaster )]
		public int CurArcaneCharges
		{
			get{ return m_CurArcaneCharges; }
			set{ m_CurArcaneCharges = value; InvalidateProperties(); Update(); }
		}

		[CommandProperty( AccessLevel.GameMaster )]
		public bool IsArcane
		{
			get{ return ( m_MaxArcaneCharges > 0 && m_CurArcaneCharges >= 0 ); }
		}

		public void Update()
		{
			if ( IsArcane )
				ItemID = 0x26AE;
			else if ( ItemID == 0x26AE )
				ItemID = 0x1F04;

			if ( IsArcane && CurArcaneCharges == 0 )
				Hue = 0;
		}

		public override void GetProperties( ObjectPropertyList list )
		{
			base.GetProperties( list );

			if ( IsArcane )
				list.Add( 1061837, "{0}\t{1}", m_CurArcaneCharges, m_MaxArcaneCharges ); // arcane charges: ~1_val~ / ~2_val~
		}

		public override void OnSingleClick( Mobile from )
		{
			base.OnSingleClick( from );

			if ( IsArcane )
				LabelTo( from, 1061837, String.Format( "{0}\t{1}", m_CurArcaneCharges, m_MaxArcaneCharges ) );
		}

		public void Flip()
		{
			if ( ItemID == 0x1F03 )
				ItemID = 0x1F04;
			else if ( ItemID == 0x1F04 )
				ItemID = 0x1F03;
		}
		#endregion
        public override int BasePhysicalResistance { get { return 2; } }
        public override int BaseFireResistance { get { return 3; } }
        public override int BaseColdResistance { get { return 4; } }
        public override int BasePoisonResistance { get { return 4; } }
        public override int BaseEnergyResistance { get { return 2; } }

        public override int InitMinHits { get { return 40; } }
        public override int InitMaxHits { get { return 50; } }

        public virtual int BaseIntBonus { get { return 1; } }

		[Constructable]
		public SilkRobe() : this( 0 )
		{
		}

		[Constructable]
		public SilkRobe( int hue ) : base( 0x1F03, hue )
		{
			Weight = 3.0;
            Name = "Silk Robe";
		}

        public SilkRobe(Serial serial)
            : base(serial)
		{
		}

		public override void Serialize( GenericWriter writer )
		{
			base.Serialize( writer );

			writer.Write( (int) 1 ); // version

			if ( IsArcane )
			{
				writer.Write( true );
				writer.Write( (int) m_CurArcaneCharges );
				writer.Write( (int) m_MaxArcaneCharges );
			}
			else
			{
				writer.Write( false );
			}
		}

		public override void Deserialize( GenericReader reader )
		{
			base.Deserialize( reader );

			int version = reader.ReadInt();

			switch ( version )
			{
				case 1:
				{
					if ( reader.ReadBool() )
					{
						m_CurArcaneCharges = reader.ReadInt();
						m_MaxArcaneCharges = reader.ReadInt();

						if ( Hue == 2118 )
							Hue = ArcaneGem.DefaultArcaneHue;
					}

					break;
				}
			}
		}
	}

	[Flipable( 0x1f01, 0x1f02 )]
	public class PlainSilkDress : BaseSilkOuterTorso
	{
        public override int BasePhysicalResistance { get { return 2; } }
        public override int BaseFireResistance { get { return 1; } }
        public override int BaseColdResistance { get { return 4; } }
        public override int BasePoisonResistance { get { return 4; } }
        public override int BaseEnergyResistance { get { return 1; } }

        public override int InitMinHits { get { return 10; } }
        public override int InitMaxHits { get { return 50; } }

		[Constructable]
		public PlainSilkDress() : this( 0 )
		{
		}

		[Constructable]
		public PlainSilkDress( int hue ) : base( 0x1F01, hue )
		{
			Weight = 2.0;
            Name = "Plain Silk Dress";
		}

        public PlainSilkDress(Serial serial)
            : base(serial)
		{
		}

		public override void Serialize( GenericWriter writer )
		{
			base.Serialize( writer );

			writer.Write( (int) 0 ); // version
		}

		public override void Deserialize( GenericReader reader )
		{
			base.Deserialize( reader );

			int version = reader.ReadInt();

			if ( Weight == 3.0 )
				Weight = 2.0;
		}
	}

	[Flipable( 0x2799, 0x27E4 )]
	public class SilkKamishimo : BaseSilkOuterTorso
	{
        public override int BasePhysicalResistance { get { return 2; } }
        public override int BaseFireResistance { get { return 1; } }
        public override int BaseColdResistance { get { return 4; } }
        public override int BasePoisonResistance { get { return 4; } }
        public override int BaseEnergyResistance { get { return 1; } }

        public override int InitMinHits { get { return 10; } }
        public override int InitMaxHits { get { return 50; } }

		[Constructable]
		public SilkKamishimo() : this( 0 )
		{
		}

		[Constructable]
		public SilkKamishimo( int hue ) : base( 0x2799, hue )
		{
			Weight = 3.0;
            Name = "Silk Kamishimo";
		}

        public SilkKamishimo(Serial serial)
            : base(serial)
		{
		}

		public override void Serialize( GenericWriter writer )
		{
			base.Serialize( writer );

			writer.Write( (int) 0 ); // version
		}

		public override void Deserialize( GenericReader reader )
		{
			base.Deserialize( reader );

			int version = reader.ReadInt();
		}
	}

	[Flipable( 0x279C, 0x27E7 )]
	public class SilkHakamaShita : BaseOuterTorso
	{
        public override int BasePhysicalResistance { get { return 2; } }
        public override int BaseFireResistance { get { return 1; } }
        public override int BaseColdResistance { get { return 4; } }
        public override int BasePoisonResistance { get { return 4; } }
        public override int BaseEnergyResistance { get { return 1; } }

        public override int InitMinHits { get { return 10; } }
        public override int InitMaxHits { get { return 50; } }

		[Constructable]
		public SilkHakamaShita() : this( 0 )
		{
		}

		[Constructable]
		public SilkHakamaShita( int hue ) : base( 0x279C, hue )
		{
			Weight = 3.0;
            Name = "Silk Hakama-Shita";
		}

        public SilkHakamaShita(Serial serial)
            : base(serial)
		{
		}

		public override void Serialize( GenericWriter writer )
		{
			base.Serialize( writer );

			writer.Write( (int) 0 ); // version
		}

		public override void Deserialize( GenericReader reader )
		{
			base.Deserialize( reader );

			int version = reader.ReadInt();
		}
	}

	[Flipable( 0x2782, 0x27CD )]
	public class SilkMaleKimono : BaseSilkOuterTorso
	{
        public override int BasePhysicalResistance { get { return 2; } }
        public override int BaseFireResistance { get { return 1; } }
        public override int BaseColdResistance { get { return 4; } }
        public override int BasePoisonResistance { get { return 4; } }
        public override int BaseEnergyResistance { get { return 1; } }

        public override int InitMinHits { get { return 10; } }
        public override int InitMaxHits { get { return 50; } }

		[Constructable]
		public SilkMaleKimono() : this( 0 )
		{
		}

		[Constructable]
		public SilkMaleKimono( int hue ) : base( 0x2782, hue )
		{
			Weight = 3.0;
            Name = "Silk Male Kimono";
		}

		public override bool AllowFemaleWearer{ get{ return false; } }

        public SilkMaleKimono(Serial serial)
            : base(serial)
		{
		}

		public override void Serialize( GenericWriter writer )
		{
			base.Serialize( writer );

			writer.Write( (int) 0 ); // version
		}

		public override void Deserialize( GenericReader reader )
		{
			base.Deserialize( reader );

			int version = reader.ReadInt();
		}
	}

	[Flipable( 0x2783, 0x27CE )]
	public class SilkFemaleKimono : BaseSilkOuterTorso
	{
        public override int BasePhysicalResistance { get { return 2; } }
        public override int BaseFireResistance { get { return 1; } }
        public override int BaseColdResistance { get { return 4; } }
        public override int BasePoisonResistance { get { return 4; } }
        public override int BaseEnergyResistance { get { return 1; } }

        public override int InitMinHits { get { return 10; } }
        public override int InitMaxHits { get { return 50; } }

		[Constructable]
		public SilkFemaleKimono() : this( 0 )
		{
		}

		[Constructable]
		public SilkFemaleKimono( int hue ) : base( 0x2783, hue )
		{
			Weight = 3.0;
            Name = "Silk Female Kimono";
		}

		public override bool AllowMaleWearer{ get{ return false; } }

        public SilkFemaleKimono(Serial serial)
            : base(serial)
		{
		}

		public override void Serialize( GenericWriter writer )
		{
			base.Serialize( writer );

			writer.Write( (int) 0 ); // version
		}

		public override void Deserialize( GenericReader reader )
		{
			base.Deserialize( reader );

			int version = reader.ReadInt();
		}
	}

	[Flipable( 0x2FB9, 0x3173 )]
	public class SilkMaleElvenRobe : BaseSilkOuterTorso
	{
        public override int BasePhysicalResistance { get { return 2; } }
        public override int BaseFireResistance { get { return 1; } }
        public override int BaseColdResistance { get { return 4; } }
        public override int BasePoisonResistance { get { return 4; } }
        public override int BaseEnergyResistance { get { return 1; } }

        public override int InitMinHits { get { return 10; } }
        public override int InitMaxHits { get { return 50; } }

		// TODO: Supports arcane?
		public override Race RequiredRace { get { return Race.Elf; } }

		[Constructable]
		public SilkMaleElvenRobe() : this( 0 )
		{
		}

		[Constructable]
		public SilkMaleElvenRobe( int hue ) : base( 0x2FB9, hue )
		{
			Weight = 2.0;
            Name = "Male Elven Silk Robe";
		}

		public override bool AllowFemaleWearer{ get{ return false; } }

        public SilkMaleElvenRobe(Serial serial)
            : base(serial)
		{
		}

		public override void Serialize( GenericWriter writer )
		{
			base.Serialize( writer );

			writer.WriteEncodedInt( 0 ); // version
		}

		public override void Deserialize( GenericReader reader )
		{
			base.Deserialize( reader );

			int version = reader.ReadEncodedInt();
		}
	}

	[Flipable( 0x2FBA, 0x3174 )]
	public class SilkFemaleElvenRobe : BaseOuterTorso
	{
        public override int BasePhysicalResistance { get { return 2; } }
        public override int BaseFireResistance { get { return 1; } }
        public override int BaseColdResistance { get { return 4; } }
        public override int BasePoisonResistance { get { return 4; } }
        public override int BaseEnergyResistance { get { return 1; } }

        public override int InitMinHits { get { return 10; } }
        public override int InitMaxHits { get { return 50; } }
		// TODO: Supports arcane?
		public override Race RequiredRace { get { return Race.Elf; } }
		[Constructable]
		public SilkFemaleElvenRobe() : this( 0 )
		{
		}

		[Constructable]
		public SilkFemaleElvenRobe( int hue ) : base( 0x2FBA, hue )
		{
			Weight = 2.0;
            Name = "Female Elven Silk Robe";
		}

		public override bool AllowMaleWearer{ get{ return false; } }

        public SilkFemaleElvenRobe(Serial serial)
            : base(serial)
		{
		}

		public override void Serialize( GenericWriter writer )
		{
			base.Serialize( writer );

			writer.WriteEncodedInt( 0 ); // version
		}

		public override void Deserialize( GenericReader reader )
		{
			base.Deserialize( reader );

			int version = reader.ReadEncodedInt();
		}
	}
}